import { getDictionary, Locale } from "@/lib/i18n";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import { Lightbulb, Award, Handshake, Users } from "lucide-react";
import Link from "next/link";

export default async function AboutPage({
  params,
}: {
  params: { lang: Locale };
}) {
  const { lang } = params;
  const dictionary = await getDictionary(lang);
  const { about } = dictionary;

  return (
    <div className="flex flex-col min-h-screen">
      {/* Hero Section */}
      <section className="py-20 md:py-32 bg-gradient-to-br from-brand-primary/5 via-white to-brand-secondary/5">
        <div className="container px-4 md:px-6">
          <div className="max-w-3xl mx-auto text-center space-y-6">
            <h1 className="text-4xl md:text-6xl font-bold tracking-tighter bg-clip-text text-transparent bg-gradient-to-r from-brand-primary to-brand-secondary">
              {about.hero.title}
            </h1>
            <p className="text-xl text-text-secondary max-w-2xl mx-auto">
              {about.hero.subtitle}
            </p>
          </div>
        </div>
      </section>

      {/* Mission Section */}
      <section className="py-20 md:py-24">
        <div className="container px-4 md:px-6">
          <div className="grid md:grid-cols-2 gap-12 items-center max-w-6xl mx-auto">
            <div>
              <div className="flex items-center gap-4 mb-6">
                <div className="w-12 h-12 rounded-lg bg-brand-primary/10 flex items-center justify-center">
                  <Lightbulb className="w-6 h-6 text-brand-primary" />
                </div>
                <h2 className="text-3xl md:text-4xl font-bold">
                  {about.mission.title}
                </h2>
              </div>
              <p className="text-lg text-text-secondary leading-relaxed">
                {about.mission.content}
              </p>
            </div>
            <div className="relative rounded-2xl overflow-hidden shadow-xl">
              <img 
                src="https://images.unsplash.com/photo-1522071820081-009f0129c71c?w=800&h=600&fit=crop&q=80" 
                alt="Team brainstorming" 
                className="w-full h-full object-cover aspect-[4/3]"
              />
            </div>
          </div>
        </div>
      </section>

      {/* Values Section */}
      <section className="py-20 md:py-24 bg-gray-50 relative overflow-hidden">
        {/* Background image with overlay */}
        <div className="absolute inset-0 opacity-5">
          <img 
            src="https://images.unsplash.com/photo-1497366216548-37526070297c?w=1920&h=1080&fit=crop&q=80" 
            alt="Modern office" 
            className="w-full h-full object-cover"
          />
        </div>
        
        <div className="container px-4 md:px-6 relative z-10">
          <div className="max-w-5xl mx-auto">
            <h2 className="text-3xl md:text-4xl font-bold text-center mb-12">
              {about.values.title}
            </h2>
            <div className="grid md:grid-cols-3 gap-8">
              <Card className="border-2 hover:border-brand-primary transition-colors">
                <CardContent className="p-6 space-y-4">
                  <div className="w-12 h-12 rounded-lg bg-brand-primary/10 flex items-center justify-center">
                    <Lightbulb className="w-6 h-6 text-brand-primary" />
                  </div>
                  <h3 className="text-xl font-semibold">
                    {about.values.items.innovation.title}
                  </h3>
                  <p className="text-text-secondary">
                    {about.values.items.innovation.desc}
                  </p>
                </CardContent>
              </Card>

              <Card className="border-2 hover:border-brand-primary transition-colors">
                <CardContent className="p-6 space-y-4">
                  <div className="w-12 h-12 rounded-lg bg-brand-secondary/10 flex items-center justify-center">
                    <Award className="w-6 h-6 text-brand-secondary" />
                  </div>
                  <h3 className="text-xl font-semibold">
                    {about.values.items.quality.title}
                  </h3>
                  <p className="text-text-secondary">
                    {about.values.items.quality.desc}
                  </p>
                </CardContent>
              </Card>

              <Card className="border-2 hover:border-brand-primary transition-colors">
                <CardContent className="p-6 space-y-4">
                  <div className="w-12 h-12 rounded-lg bg-accent-purple/10 flex items-center justify-center">
                    <Handshake className="w-6 h-6 text-accent-purple" />
                  </div>
                  <h3 className="text-xl font-semibold">
                    {about.values.items.partnership.title}
                  </h3>
                  <p className="text-text-secondary">
                    {about.values.items.partnership.desc}
                  </p>
                </CardContent>
              </Card>
            </div>
          </div>
        </div>
      </section>

      {/* Team Section */}
      <section className="py-20 md:py-24">
        <div className="container px-4 md:px-6">
          <div className="max-w-5xl mx-auto text-center space-y-6 mb-12">
            <div className="w-16 h-16 rounded-lg bg-brand-primary/10 flex items-center justify-center mx-auto">
              <Users className="w-8 h-8 text-brand-primary" />
            </div>
            <h2 className="text-3xl md:text-4xl font-bold">
              {about.team.title}
            </h2>
            <p className="text-xl text-text-secondary max-w-2xl mx-auto">
              {about.team.subtitle}
            </p>
          </div>

          {/* Team Photos Grid */}
          <div className="grid grid-cols-2 md:grid-cols-4 gap-6 max-w-4xl mx-auto">
            <div className="relative aspect-square rounded-xl overflow-hidden shadow-lg group">
              <img 
                src="https://images.unsplash.com/photo-1560250097-0b93528c311a?w=400&h=400&fit=crop&q=80" 
                alt="Team member" 
                className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
              />
              <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity">
                <div className="absolute bottom-4 left-4 text-white">
                  <p className="font-semibold">CEO</p>
                </div>
              </div>
            </div>
            <div className="relative aspect-square rounded-xl overflow-hidden shadow-lg group">
              <img 
                src="https://images.unsplash.com/photo-1573496359142-b8d87734a5a2?w=400&h=400&fit=crop&q=80" 
                alt="Team member" 
                className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
              />
              <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity">
                <div className="absolute bottom-4 left-4 text-white">
                  <p className="font-semibold">CTO</p>
                </div>
              </div>
            </div>
            <div className="relative aspect-square rounded-xl overflow-hidden shadow-lg group">
              <img 
                src="https://images.unsplash.com/photo-1519085360753-af0119f7cbe7?w=400&h=400&fit=crop&q=80" 
                alt="Team member" 
                className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
              />
              <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity">
                <div className="absolute bottom-4 left-4 text-white">
                  <p className="font-semibold">Lead Developer</p>
                </div>
              </div>
            </div>
            <div className="relative aspect-square rounded-xl overflow-hidden shadow-lg group">
              <img 
                src="https://images.unsplash.com/photo-1580489944761-15a19d654956?w=400&h=400&fit=crop&q=80" 
                alt="Team member" 
                className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-300"
              />
              <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent opacity-0 group-hover:opacity-100 transition-opacity">
                <div className="absolute bottom-4 left-4 text-white">
                  <p className="font-semibold">Product Manager</p>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="py-20 md:py-24 bg-gradient-to-br from-brand-primary to-brand-secondary text-white">
        <div className="container px-4 md:px-6">
          <div className="max-w-3xl mx-auto text-center space-y-6">
            <h2 className="text-3xl md:text-4xl font-bold">
              {dictionary.home.cta.title}
            </h2>
            <p className="text-xl opacity-90">
              {dictionary.home.cta.desc}
            </p>
            <div className="flex flex-col sm:flex-row gap-4 justify-center pt-4">
              <Link href={`/${lang}/contact`}>
                <Button size="lg" variant="secondary" className="w-full sm:w-auto">
                  {dictionary.navigation.contact}
                </Button>
              </Link>
              <Link href={`/${lang}/#products`}>
                <Button size="lg" variant="outline" className="w-full sm:w-auto border-white text-white hover:bg-white hover:text-brand-primary">
                  {dictionary.navigation.products}
                </Button>
              </Link>
            </div>
          </div>
        </div>
      </section>
    </div>
  );
}
